#!/bin/bash
# filename: autoOptimzie.sh
# function: Optimizing onconfig for GBase 8s V8.8
# write by: liaosnet@gbasedbt.com 2024-01-26
##### Defind env
export LANG=C
_loginfo(){
  echo -e "[$(date +'%Y-%m-%d %H:%M:%S')] $*"
}

USER_NAME=gbasedbt
USER_UPPER=$(echo $USER_NAME|tr [a-z] [A-Z])
USER_HOME=$HOME
INSTALL_DIR=$(env | awk -F'=' -v dir="${USER_UPPER}DIR" '$1==dir{print $2}')
CFGFILE=${INSTALL_DIR}/etc/$ONCONFIG

WORKDIR=$(pwd)
##### Check env
if [ ! x"$(whoami)" = "x${USER_NAME}" ]; then
  echo "Must run as user: ${USER_NAME}"
  exit 1
fi

##### Get env
NUMCPU=$(awk '/^processor/{i++}END{printf("%d\n",i)}' /proc/cpuinfo)
NUMMEM=$(awk '/^MemTotal:/{printf("%d\n",$2/1000)}' /proc/meminfo)

if [ ${NUMCPU:-0} -eq 0 ]; then
  echo "GET cpu information error."
  exit 2 
elif [ $NUMCPU -le 4 ]; then
  CPUVP=$NUMCPU
  CFG_NETPOOL=1
else
  CPUVP=$(expr $NUMCPU - 1)
  CFG_NETPOOL=$(expr $NUMCPU / 3)
fi

if [ ${NUMMEM:-0} -eq 0 ]; then
  echo "GET memory information error."
  exit 2
elif [ $NUMMEM -le 2048 ]; then
  # mem less then 1G, use direct_io, only 2k buffpool
  CFG_DIRECT_IO=1
  CFG_LOCKS=50000
  CFG_SHMVIRTSIZE=384000
  CFG_2KPOOL=20000
  CFG_16KPOOL=10000 
elif [ $NUMMEM -le 4096 ]; then
  # mem less then 4G, use direct_io, only 2k buffpool
  CFG_DIRECT_IO=1
  CFG_LOCKS=200000
  CFG_SHMVIRTSIZE=512000
  CFG_2KPOOL=50000
  CFG_16KPOOL=20000 
elif [ $NUMMEM -le 8192 ]; then
  # mem less then 8G, use direct_io, only 2k buffpool
  CFG_DIRECT_IO=1
  MUTI=$(expr $NUMMEM / 2000)
  [ $MUTI -eq 0 ] && MUTI=1
  CFG_LOCKS=1000000
  CFG_SHMVIRTSIZE=512000
  CFG_2KPOOL=500000
  CFG_16KPOOL=100000
elif [ $NUMMEM -le 32768 ]; then
  # mem >8G && < 32G, not use direct_io, use 2k & 16k buffpool
  CFG_DIRECT_IO=0
  MUTI=$(expr $NUMMEM / 8000)
  [ $MUTI -eq 0 ] && MUTI=2
  CFG_LOCKS=5000000
  CFG_SHMVIRTSIZE=$(awk -v n="$MUTI" 'BEGIN{print (n-1)*1024000}')
  CFG_2KPOOL=500000
  CFG_16KPOOL=$(awk -v n="$MUTI" 'BEGIN{print (n-1)*200000}')
else
  # mem > 32G
  CFG_DIRECT_IO=0
  CFG_LOCKS=5000000
  CFG_SHMVIRTSIZE=4096000
  CFG_2KPOOL=1000000
  CFG_16KPOOL=1000000
fi
CFG_SHMADD=$(expr ${CFG_SHMVIRTSIZE:-1024000} / 4)
CFG_SHMTOTAL=$(expr $NUMMEM \* 900)

# dynamic value
_loginfo "Optimizing database config."
sed -i "s#^NETTYPE.*#NETTYPE soctcp,${CFG_NETPOOL},200,NET#g" $CFGFILE
sed -i "s#^VPCLASS.*cpu.*#VPCLASS cpu,num=${CPUVP},noage#g" $CFGFILE
sed -i "s#^DIRECT_IO.*#DIRECT_IO 0#g" $CFGFILE
sed -i "s#^LOCKS.*#LOCKS ${CFG_LOCKS}#g" $CFGFILE
sed -i "s#^SHMVIRTSIZE.*#SHMVIRTSIZE ${CFG_SHMVIRTSIZE}#g" $CFGFILE
sed -i "s#^SHMADD.*#SHMADD ${CFG_SHMADD}#g" $CFGFILE
sed -i "s#^SHMTOTAL.*#SHMTOTAL ${CFG_SHMTOTAL}#g" $CFGFILE
sed -i "s#^OPT_GOAL.*#OPT_GOAL 0#g" $CFGFILE
sed -i "s#^TAPEBLK.*#TAPEBLK 512#g" $CFGFILE
sed -i "s#^LTAPEBLK.*#LTAPEBLK 512#g" $CFGFILE

sed -i "s#^BUFFERPOOL.*size=2.*#BUFFERPOOL size=2K,buffers=${CFG_2KPOOL},lrus=8,lru_min_dirty=50,lru_max_dirty=60#g" $CFGFILE
sed -i "s#^BUFFERPOOL.*size=16.*#BUFFERPOOL size=16K,buffers=${CFG_16KPOOL},lrus=64,lru_min_dirty=50,lru_max_dirty=60#g" $CFGFILE

_loginfo "Finish optimize config, Plase RESTART database!"

exit 0
