#!/bin/bash
##################################################################
# Filename: AutoInit_GBase8s.sh
# Function: Auto install GBase 8s software and auto init database.
# Write by: liaosnet@gbasedbt.com
# Version : 1.5.0   update date: 2022-09-24
##################################################################
##### Defind env
export LANG=C
loginfo(){
  echo -e "[$(date +'%Y-%m-%d %H:%M:%S')] $*"
}
##### Get Parameter
while [[ $# -gt 0 ]]
do
  key="$1"
  case $key in
    -d) DATADIR="$2";     shift 2 ;;
    -i) INSTALL_DIR="$2"; shift 2 ;;
    -p) USER_HOME="$2";   shift 2 ;;
    -s) DBS1GB="$2";      shift 2 ;;
    -l) GBASELOCALE="$2"; shift 2 ;;
    -o) SOFTONLY="$2";    shift 2 ;;
    -u) USER_NAME="$2";   shift 2 ;;
    -n) GBASESERVER="$2"; shift 2 ;;
    *)
        cat <<!
Usage:
    AutoInit_GBase8s.sh [-d path] [-i path] [-p path] [-s y|n] [-l locale] [-u user] [-o y|n] [-n SERVNAME]

        -d path     The path of dbspace.
        -i path     The path of install software.
        -p path     The path of home path.
        -s y|n      Value of dbspace is 1GB? Yes/No, default is N.
        -u user     The user name for SYSDBA, gbasedbt/informix, default is gbasedbt
        -l locale   DB_LOCALE/CLIENT_LOCALE value.
        -o y|n      Only install software? Yes/No, default is N.
        -n SERVNAME User define GBASEDBTSERVER.

!
        exit 1
        ;;
  esac
done
##### Define Parameter
USER_NAME=${USER_NAME:-gbasedbt}
if [ x"$USER_NAME" = "xgbasedbt" -o x"$USER_NAME" = "xinformix" ]; then
   loginfo "The SYSDBA user is: $USER_NAME"
else
   echo "Username error!"
   exit 1
fi
USER_UPPER=$(echo $USER_NAME|tr [a-z] [A-Z])
USER_HOME=${USER_HOME:-/home/gbase}
USER_PASS=GBase123
INSTALL_DIR=${INSTALL_DIR:-/opt/gbase}
GBASESERVER=${GBASESERVER:-gbase01}
GBASELOCALE=${GBASELOCALE:-zh_CN.utf8}
PORTNO=9088

DATADIR=${DATADIR:-/data/gbase}
BACKUPDIR=/data/backup
### dbspace init size.
DBS1GB=${DBS1GB:-y}
ROOTSIZE=1024000
PLOGSIZE=2048000
LLOGSIZE=4096000
SBSPACESIZE=4096000
TEMPSIZE=4096000
DATASIZE=10240000

if [ x"$DBS1GB" = xy -o x"$DBS1GB" = xY ]; then
  ROOTSIZE=1024000
  PLOGSIZE=1024000
  LLOGSIZE=1024000
  SBSPACESIZE=1024000
  TEMPSIZE=1024000
  DATASIZE=1024000
fi

WORKDIR=$(pwd)
##### Check env
if [ ! x"$(whoami)" = "xroot" ]; then
  echo "Must run as user: root"
  exit 1
fi 
if [ x"${USER_HOME}" = x"${INSTALL_DIR}" ]; then
  INSTALL_DIR=${USER_HOME}/gbase
fi
if [ -d ${INSTALL_DIR} ] && [ ! x"$(ls -A ${INSTALL_DIR})" = x ]; then
  INSTALL_DIR=${INSTALL_DIR}/Server
fi
if [ -d ${DATADIR} ] && [ ! x"$(ls -A ${DATADIR})" = x ]; then
  DATADIR=${INSTALL_DIR}/data
fi

SOFTPACKNAME=$(ls GBase*.tar 2>/dev/null)
CSDKPACKNAME=$(ls clientsdk_*.tar 2>/dev/null)
CERTPACKNAME=$(ls gbasedbt_certs.tar 2>/dev/null)
if [ x"$SOFTPACKNAME" = x ]; then
  loginfo "ERROR: GBase Database Soft Package not found."
  exit 2
fi

### CheckEnv.sh
bash CheckEnv.sh
if [ $? -gt 0 ]; then
    exit 3
fi

# IP use first IPADDR
# IPADDR=$(ifconfig -a | awk '/inet /{printf (split($2,a,":")>1)?a[2]:$2" "}')
loginfo "IPADDR: 0.0.0.0"
loginfo "Datadir: $DATADIR"

##### Get env
NUMCPU=$(awk '/^processor/{i++}END{printf("%d\n",i)}' /proc/cpuinfo)
NUMMEM=$(awk '/^MemTotal:/{printf("%d\n",$2/1000)}' /proc/meminfo)

if [ ${NUMCPU:-0} -eq 0 ]; then
  echo "GET cpu information error."
  exit 2 
elif [ $NUMCPU -le 4 ]; then
  CPUVP=$NUMCPU
  CFG_NETPOOL=1
else
  CPUVP=$(expr $NUMCPU - 1)
  CFG_NETPOOL=$(expr $NUMCPU / 3)
fi

if [ ${NUMMEM:-0} -eq 0 ]; then
  echo "GET memory information error."
  exit 2
elif [ $NUMMEM -le 2048 ]; then
  # mem less then 1G, use direct_io, only 2k buffpool
  CFG_DIRECT_IO=1
  CFG_LOCKS=50000
  CFG_SHMVIRTSIZE=384000
  CFG_2KPOOL=20000
  CFG_16KPOOL=10000 
elif [ $NUMMEM -le 4096 ]; then
  # mem less then 4G, use direct_io, only 2k buffpool
  CFG_DIRECT_IO=1
  CFG_LOCKS=200000
  CFG_SHMVIRTSIZE=512000
  CFG_2KPOOL=50000
  CFG_16KPOOL=20000 
elif [ $NUMMEM -le 8192 ]; then
  # mem less then 8G, use direct_io, only 2k buffpool
  CFG_DIRECT_IO=1
  MUTI=$(expr $NUMMEM / 2000)
  [ $MUTI -eq 0 ] && MUTI=1
  CFG_LOCKS=1000000
  CFG_SHMVIRTSIZE=512000
  CFG_2KPOOL=500000
  CFG_16KPOOL=100000
elif [ $NUMMEM -le 32768 ]; then
  # mem >8G && < 32G, not use direct_io, use 2k & 16k buffpool
  CFG_DIRECT_IO=0
  MUTI=$(expr $NUMMEM / 8000)
  [ $MUTI -eq 0 ] && MUTI=2
  CFG_LOCKS=5000000
  CFG_SHMVIRTSIZE=$(awk -v n="$MUTI" 'BEGIN{print (n-1)*1024000}')
  CFG_2KPOOL=500000
  CFG_16KPOOL=$(awk -v n="$MUTI" 'BEGIN{print (n-1)*200000}')
else
  # mem > 32G
  CFG_DIRECT_IO=0
  CFG_LOCKS=5000000
  CFG_SHMVIRTSIZE=4096000
  CFG_2KPOOL=1000000
  CFG_16KPOOL=1000000
fi
PAGESIZE="-k 16"
CFG_SHMADD=$(expr ${CFG_SHMVIRTSIZE:-1024000} / 4)
CFG_SHMTOTAL=$(expr $NUMMEM \* 900)

##### Create group and user
id ${USER_NAME} >/dev/null 2>&1
if [ $? -eq 0 ];then
    echo "User: ${USER_NAME} exists, you must check !" 
    exit 2
fi
loginfo "Creating group [${USER_NAME}] and user [${USER_NAME}] with HOME [$USER_HOME]."
groupadd -g 2000 ${USER_NAME} 2>/dev/null
if [ $? -gt 0 ]; then
  echo "Create group [${USER_NAME}] error."
  exit 3
fi
useradd -u 2000 -g ${USER_NAME} -d ${USER_HOME:-/home/${USER_NAME}} -m -s /bin/bash ${USER_NAME} 2>/dev/null
if [ $? -gt 0 ]; then
  echo "Create user [${USER_NAME}] error."
  exit 3
fi
loginfo "Creating group dbaao and user dbaao with HOME /home/dbaao."
groupadd -g 2001 dbaao 2>/dev/null
if [ $? -gt 0 ]; then
  echo "Create group dbaao error."
  exit 3
fi
useradd -u 2001 -g dbaao -d /home/dbaao -m -s /bin/bash dbaao 2>/dev/null
if [ $? -gt 0 ]; then
  echo "Create user dbaao error."
  exit 3
fi

loginfo "Creating group dbsso and user dbsso with HOME /home/dbsso."
groupadd -g 2002 dbsso 2>/dev/null
if [ $? -gt 0 ]; then
  echo "Create group dbsso error."
  exit 3
fi
useradd -u 2002 -g dbsso -d /home/dbsso -m -s /bin/bash dbsso 2>/dev/null
if [ $? -gt 0 ]; then
  echo "Create user dbsso error."
  exit 3
fi

loginfo "Seting password for user ${USER_NAME}, dbaao and dbsso."
echo "${USER_NAME}:${USER_PASS}" | chpasswd >/dev/null 2>&1
echo "dbaao:${USER_PASS}" | chpasswd >/dev/null 2>&1
echo "dbsso:${USER_PASS}" | chpasswd >/dev/null 2>&1

mkdir -p ${USER_HOME}/users 2>/dev/null
chmod 0755 ${USER_HOME} 2>/dev/null
chown ${USER_NAME}:${USER_NAME} ${USER_HOME} 2>/dev/null
chmod 0777 ${USER_HOME}/users 2>/dev/null
chown ${USER_NAME}:${USER_NAME} ${USER_HOME}/users 2>/dev/null
chmod 0750 /home/dbaao 2>/dev/null
chmod 0750 /home/dbsso 2>/dev/null

##### Unzip software and install
loginfo "Unziping [${SOFTPACKNAME}]."
mkdir -p ${WORKDIR}/install 2>/dev/null
cd ${WORKDIR}/install
tar -xf ${WORKDIR}/${SOFTPACKNAME} 2>/dev/null
if [ ! -x "ids_install" ]; then
  echo "ids_install not exists."
  exit 4
fi
if [ -x "onsecurity" ]; then
  loginfo "Check path INSTALL_DIR($INSTALL_DIR) security."
  ${WORKDIR}/install/onsecurity $INSTALL_DIR >/dev/null 2>&1
  if [ $? -gt 0 ]; then
    echo "INSTALL_DIR: $INSTALL_DIR not security, Plase Check."
    exit 5
  fi
fi

mkdir -p $INSTALL_DIR 2>/dev/null
chown ${USER_NAME}:${USER_NAME} $INSTALL_DIR 2>/dev/null
chmod 0755 $INSTALL_DIR 2>/dev/null

# ids_install
loginfo "Execute software install, this will take a moment."
timeout 1800 ${WORKDIR}/install/ids_install -i silent -DLICENSE_ACCEPTED=TRUE -DUSER_INSTALL_DIR=${INSTALL_DIR:-/opt/${USER_NAME}}

cd ${WORKDIR}
if [ ! x"$CSDKPACKNAME" = x ]; then
  mkdir -p ${WORKDIR}/csdk 2>/dev/null
  cd ${WORKDIR}/csdk
  loginfo "Unziping [${CSDKPACKNAME}]."
  tar -xf ${WORKDIR}/${CSDKPACKNAME} 2>/dev/null
  timeout 1800 ${WORKDIR}/csdk/installclientsdk -i silent -DLICENSE_ACCEPTED=TRUE -DUSER_INSTALL_DIR=${INSTALL_DIR:-/opt/${USER_NAME}}
  # root use cp = 'cp -i'
  \cp ${WORKDIR}/install/.gbase.properties $INSTALL_DIR/bin/configuration/
fi
cd ${WORKDIR}

###### Init database
[ ! -d /etc/${USER_NAME} ] && mkdir -p /etc/${USER_NAME} 2>/dev/null
echo "USERS:daemon" > /etc/${USER_NAME}/allowed.surrogates

cd ${INSTALL_DIR}
chgrp -R dbaao aaodir 2>/dev/null
chgrp -R dbsso dbssodir 2>/dev/null
echo "IXUSERS=*" > ${INSTALL_DIR}/dbssodir/seccfg
chown ${USER_PASS}:dbsso ${INSTALL_DIR}/dbssodir/seccfg 2>/dev/null

mkdir -p /home/dbaao/aaodir 2>/dev/null
chmod 0755 /home/dbaao/aaodir 2>/dev/null
chown dbaao:dbaao /home/dbaao/aaodir 2>/dev/null

# profile
loginfo "Building ~${USER_NAME}/.bash_profile ."
cat >> $USER_HOME/.bash_profile <<EOF 2>/dev/null
export ${USER_UPPER}DIR=${INSTALL_DIR}
export ${USER_UPPER}SERVER=${GBASESERVER}
export ONCONFIG=onconfig.\$${USER_UPPER}SERVER
export PATH=\$${USER_UPPER}DIR/bin:\${PATH}
export ${USER_UPPER}SQLHOSTS=${INSTALL_DIR}/etc/sqlhosts

export DB_LOCALE=${GBASELOCALE}
export CLIENT_LOCALE=${GBASELOCALE}
export GL_USEGLU=1
# exprot NODEFDAC=yes
export DBDATE="Y4MD-"
export DBACCESS_SHOW_TIME=1
EOF

loginfo "Building ~dbaao/.bash_profile ."
cp $USER_HOME/.bash_profile ~dbaao/.bash_profile
chown dbaao:dbaao ~dbaao/.bash_profile

loginfo "Building ~dbsso/.bash_profile ."
cp $USER_HOME/.bash_profile ~dbsso/.bash_profile
chown dbsso:dbsso ~dbsso/.bash_profile

# sqlhosts
loginfo "Building ${INSTALL_DIR}/etc/sqlhosts ."
# echo "$GBASESERVER onsoctcp 0.0.0.0 ${PORTNO:-9088}" > $INSTALL_DIR/etc/sqlhosts
cat > $INSTALL_DIR/etc/sqlhosts <<EOF 2>/dev/null
${GBASESERVER}     onsoctcp 0.0.0.0 ${PORTNO:-9088}
${GBASESERVER}_ssl onsocssl 0.0.0.0 9089
${GBASESERVER}_shm onipcshm 0.0.0.0 9087
EOF
chown ${USER_NAME}:${USER_NAME} $INSTALL_DIR/etc/sqlhosts 2>/dev/null
chmod 0644 $INSTALL_DIR/etc/sqlhosts 2>/dev/null

# onconfig
CFGFILE=$INSTALL_DIR/etc/onconfig.$GBASESERVER
cp $INSTALL_DIR/etc/onconfig.std $CFGFILE
chown ${USER_NAME}:${USER_NAME} $CFGFILE 2>/dev/null
chmod 0644 $CFGFILE
loginfo "Building $CFGFILE ."

sed -i "s#^ROOTPATH.*#ROOTPATH $DATADIR/rootchk#g" $CFGFILE
sed -i "s#^ROOTSIZE.*#ROOTSIZE $ROOTSIZE#g" $CFGFILE
sed -i "s#^DBSERVERNAME.*#DBSERVERNAME $GBASESERVER#g" $CFGFILE
sed -i "s#^TAPEDEV.*#TAPEDEV /data/backup/data#g" $CFGFILE
sed -i "s#^LTAPEDEV.*#LTAPEDEV /dev/null#g" $CFGFILE
sed -i "s#^USERMAPPING.*#USERMAPPING ADMIN#g" $CFGFILE
sed -i "s#^DEF_TABLE_LOCKMODE.*#DEF_TABLE_LOCKMODE row#g" $CFGFILE

chown ${USER_NAME}:${USER_NAME} $CFGFILE

# datadir
loginfo "Creating DATADIR: ${DATADIR} ."
mkdir -p $DATADIR 2>/dev/null
loginfo "Change permission for directory: $DATADIR ."
chown ${USER_NAME}:${USER_NAME} $DATADIR 2>/dev/null

mkdir -p ${BACKUPDIR}/data 2>/dev/null
mkdir -p ${BACKUPDIR}/logs 2>/dev/null
chown ${USER_NAME}:${USER_NAME} ${BACKUPDIR}/data 2>/dev/null
chown ${USER_NAME}:${USER_NAME} ${BACKUPDIR}/logs 2>/dev/null
chmod 755 ${BACKUPDIR} 2>/dev/null
chmod 775 ${BACKUPDIR}/data 2>/dev/null
chmod 775 ${BACKUPDIR}/logs 2>/dev/null

DTDIR=$DATADIR
while true
do
  DTTMP="${DTDIR%/*}"
  DTDIR=$DTTMP
  if [ x"${DTDIR}" = "x" ]; then
    break
  else
    loginfo "Change permission for directory: $DTDIR ."
    chmod 0755 $DTDIR
  fi
  sleep 1
done

TMPDIR=$(pwd)
cd $DATADIR
> rootchk
touch plogchk llogchk sbspace01 tempchk01 datachk01
chown ${USER_NAME}:${USER_NAME} rootchk plogchk llogchk sbspace01 tempchk01 datachk01
chmod 0660 rootchk plogchk llogchk sbspace01 tempchk01 datachk01
cd $TMPDIR

if [ "x$SOFTONLY" = xy -o "x$SOFTONLY" = xY ];then
  loginfo "GBase 8s Database software install finished."
  cat <<!

  You MUST manually initialize the database!

!
  exit 0;
fi

# oninit
loginfo "Start run database init: oninit -ivy"
su - ${USER_NAME} -c "timeout 1800 oninit -ivy"

echo -e "OK"
loginfo "Creating system database.\c"
for w in {1..5}
do
  sleep 3
  echo -e ".\c"
done

while :
do
  sleep 3
  echo -e ".\c"
  NUMDB=$(su - ${USER_NAME} -c 'echo "select count(*) numdb from sysdatabases;" | dbaccess sysmaster - 2>/dev/null')
  NUMDB=$(echo $NUMDB | awk '{printf("%d\n",$2)}')
  if [ ${NUMDB} -gt 3 ]; then
    break
  else
    sleep 3
    echo -e ".\c"
  fi
done

## create dbspace.
echo -e ""
loginfo "Creating dbspace plogdbs."
su - ${USER_NAME} -c "onspaces -c -d plogdbs -p $DATADIR/plogchk -o 0 -s $PLOGSIZE >/dev/null 2>&1"

loginfo "Creating dbspace llogdbs."
su - ${USER_NAME} -c "onspaces -c -d llogdbs -p $DATADIR/llogchk -o 0 -s $LLOGSIZE >/dev/null 2>&1"

loginfo "Creating dbspace tempdbs01"
su - ${USER_NAME} -c "onspaces -c -d tempdbs01 -p $DATADIR/tempchk01 -t -o 0 -s $TEMPSIZE $PAGESIZE >/dev/null 2>&1"

loginfo "Creating smart blob space sbspace01"
su - ${USER_NAME} -c "onspaces -c -S sbspace01 -p $DATADIR/sbspace01 -o 0 -s $SBSPACESIZE >/dev/null 2>&1"

loginfo "Creating dbspace datadbs01"
su - ${USER_NAME} -c "onspaces -c -d datadbs01 -p $DATADIR/datachk01 -o 0 -s $DATASIZE $PAGESIZE >/dev/null 2>&1"

## change chunk extend able on
loginfo "Changing auto extend able on for chunk datadbs01"
ADMIN_SQLFILE=${INSTALL_DIR}/temp/admin_sqlfile.sql
mkdir -p ${INSTALL_DIR}/temp
cat << ! > $ADMIN_SQLFILE 2>&1
EXECUTE FUNCTION task ("modify chunk extendable on", 6);
!
if [ -s $ADMIN_SQLFILE ]; then
  chown ${USER_NAME}:${USER_NAME} ${INSTALL_DIR}/temp
  chown ${USER_NAME}:${USER_NAME} $ADMIN_SQLFILE
  su - ${USER_NAME} -c "dbaccess sysadmin $ADMIN_SQLFILE >/dev/null 2>&1"
fi

## create mapping default user
loginfo "Creating default user for mapping user"
ADMIN_SQLFILE=${INSTALL_DIR}/temp/mapping_sqlfile.sql
mkdir -p ${INSTALL_DIR}/temp
cat << ! > $ADMIN_SQLFILE 2>&1
CREATE DEFAULT USER WITH PROPERTIES USER daemon HOME "${USER_HOME}/users";
!
if [ -s $ADMIN_SQLFILE ]; then
  chown ${USER_NAME}:${USER_NAME} ${INSTALL_DIR}/temp
  chown ${USER_NAME}:${USER_NAME} $ADMIN_SQLFILE
  su - ${USER_NAME} -c "dbaccess sysuser $ADMIN_SQLFILE >/dev/null 2>&1"
fi

## physical log & logical log
loginfo "Moving physical log to plogdbs."
PLOGFILE=$(echo $PLOGSIZE | awk '{printf("%d\n",substr($1,1,1) * 10 ^ (length($1) - 1))}')
su - ${USER_NAME} -c "onparams -p -d plogdbs -s $PLOGFILE -y >/dev/null 2>&1"

loginfo "Adding 10 logical log file in llogdbs."
LLOGFILE=$(echo $LLOGSIZE | awk '{printf("%d\n",substr($1,1,1) * 10 ^ (length($1) - 1))}')
NEWFILE=$(expr $LLOGFILE / 10)
[ $NEWFILE -gt 1000000 ] && NEWFILE=1000000
for w in {1..10}
do
  su - ${USER_NAME} -c "onparams -a -d llogdbs -s $NEWFILE >/dev/null 2>&1"
done

loginfo "Moving CURRENT logical log to new logical file."
while :
do
  CURRLOG=$(su - ${USER_NAME} -c "onmode -l; onmode -c; onstat -l" | awk '/U---C-L/{print $2}')
  if [ $CURRLOG -gt 6 ]; then
    break
  else
    sleep 1
  fi
done

loginfo "Droping logical log file which in rootdbs."
for e in {1..6}
do
  su - ${USER_NAME} -c "onparams -d -l $e -y >/dev/null 2>&1"
done

### Build default database: testdb
loginfo "Create database testdb."
CRDB_SQLFILE=${INSTALL_DIR}/temp/crdb_sqlfile.sql
mkdir -p ${INSTALL_DIR}/temp
cat << ! > $CRDB_SQLFILE 2>&1
create database testdb in datadbs01 with log;
!
if [ -s $CRDB_SQLFILE ]; then
  chown ${USER_NAME}:${USER_NAME} ${INSTALL_DIR}/temp
  chown ${USER_NAME}:${USER_NAME} $CRDB_SQLFILE
  su - ${USER_NAME} -c "dbaccess sysmaster $CRDB_SQLFILE >/dev/null 2>&1"
fi

###### change config
loginfo "Creating file \$INSTALL_DIR/etc/sysadmin/stop ."
if [ ! -f $INSTALL_DIR/etc/sysadmin/stop ]; then
  touch $INSTALL_DIR/etc/sysadmin/stop
  chown ${USER_NAME}:${USER_NAME} $INSTALL_DIR/etc/sysadmin/stop
  chmod 0644 $INSTALL_DIR/etc/sysadmin/stop
fi

## cfg with static value
loginfo "Optimizing database config."
sed -i "s#^PHYSBUFF.*#PHYSBUFF 1024#g" $CFGFILE
sed -i "s#^LOGBUFF.*#LOGBUFF 1024#g" $CFGFILE
sed -i "s#^DYNAMIC_LOGS.*#DYNAMIC_LOGS 0#g" $CFGFILE
sed -i "s#^LTXHWM.*#LTXHWM 40#g" $CFGFILE
sed -i "s#^LTXEHWM.*#LTXEHWM 50#g" $CFGFILE
sed -i "s#^DBSPACETEMP.*#DBSPACETEMP tempdbs01#g" $CFGFILE
sed -i "s#^SBSPACENAME.*#SBSPACENAME sbspace01#g" $CFGFILE
sed -i "s#^SYSSBSPACENAME.*#SYSSBSPACENAME sbspace01#g" $CFGFILE
sed -i "s#^MULTIPROCESSOR.*#MULTIPROCESSOR 1#g" $CFGFILE
sed -i "s#^AUTO_TUNE.*#AUTO_TUNE 0#g" $CFGFILE
sed -i "/^AUTO_TUNE/a\AUTO_REPREPARE 1" $CFGFILE
sed -i "s#^CLEANERS.*#CLEANERS 32#g" $CFGFILE
sed -i "s#^STACKSIZE.*#STACKSIZE 512#g" $CFGFILE
sed -i "s#^ALLOW_NEWLINE.*#ALLOW_NEWLINE 1#g" $CFGFILE
sed -i "s#^DS_MAX_QUERIES.*#DS_MAX_QUERIES 5#g" $CFGFILE
sed -i "s#^DS_TOTAL_MEMORY.*#DS_TOTAL_MEMORY 1024000#g" $CFGFILE
sed -i "s#^DS_NONPDQ_QUERY_MEM.*#DS_NONPDQ_QUERY_MEM 256000#g" $CFGFILE
sed -i "s#^TEMPTAB_NOLOG.*#TEMPTAB_NOLOG 1#g" $CFGFILE
sed -i "s#^DUMPSHMEM.*#DUMPSHMEM 0#g" $CFGFILE
sed -i 's#^ALARMPROGRAM.*#ALARMPROGRAM $GBASEDBTDIR/etc/log_full.sh#g' $CFGFILE
sed -i "s#^LTAPEDEV.*#LTAPEDEV /data/backup/logs#g" $CFGFILE
sed -i "s#^DBSERVERALIASES.*#DBSERVERALIASES ${GBASESERVER}_ssl,${GBASESERVER}_shm#g" $CFGFILE
# sed -i "s#^IFX_FOLDVIEW.*#IFX_FOLDVIEW 0#g" $CFGFILE

if [ $NUMMEM -le 4096 ]; then
  sed -i "s#^DS_TOTAL_MEMORY.*#DS_TOTAL_MEMORY 128000#g" $CFGFILE
  sed -i "s#^DS_NONPDQ_QUERY_MEM.*#DS_NONPDQ_QUERY_MEM 32000#g" $CFGFILE
fi
# dynamic value
sed -i "s#^NETTYPE.*#NETTYPE soctcp,${CFG_NETPOOL},200,NET#g" $CFGFILE
sed -i "/^NETTYPE/a\NETTYPE socssl,${CFG_NETPOOL},200,NET" $CFGFILE
sed -i "s#^VPCLASS.*cpu.*#VPCLASS cpu,num=${CPUVP},noage#g" $CFGFILE
sed -i "/^VPCLASS/a\VPCLASS encrypt,num=${CFG_NETPOOL}" $CFGFILE
sed -i "s#^DIRECT_IO.*#DIRECT_IO ${CFG_DIRECT_IO}#g" $CFGFILE
sed -i "s#^LOCKS.*#LOCKS ${CFG_LOCKS}#g" $CFGFILE
sed -i "s#^SHMVIRTSIZE.*#SHMVIRTSIZE ${CFG_SHMVIRTSIZE}#g" $CFGFILE
sed -i "s#^SHMADD.*#SHMADD ${CFG_SHMADD}#g" $CFGFILE
sed -i "s#^SHMTOTAL.*#SHMTOTAL ${CFG_SHMTOTAL}#g" $CFGFILE

sed -i "s#^BUFFERPOOL.*size=2.*#BUFFERPOOL size=2K,buffers=${CFG_2KPOOL},lrus=8,lru_min_dirty=50,lru_max_dirty=60#g" $CFGFILE
sed -i "s#^BUFFERPOOL.*size=16.*#BUFFERPOOL size=16K,buffers=${CFG_16KPOOL},lrus=64,lru_min_dirty=50,lru_max_dirty=60#g" $CFGFILE

sed -i 's#^BACKUP_CMD.*#BACKUP_CMD="ontape -a -d"#g' ${INSTALL_DIR}/etc/log_full.sh
loginfo "Install certs for SSL connection."
if [ ! x"${CERTPACKNAME}" = x ]; then
  cd ${WORKDIR}
  tar -xf gbasedbt_certs.tar -C ${INSTALL_DIR} 2>/dev/null
  chown -R ${USER_NAME}:${USER_NAME} ${INSTALL_DIR}/certs 2>/dev/null
fi
loginfo "Building ${INSTALL_DIR}/etc/sqlhosts ."
cat << ! > ${INSTALL_DIR}/etc/sqlhosts.ext 2>/dev/null
[config]
GBS_TYPE=gbasessl

# DBSERVERNAME or DBSERVERALIASES with onsocssl
[${GBASESERVER}_ssl]
server=gbase_ssl_server
client=gbase_ssl_client

[gbase_ssl_server]
TLSCACertificateFile=${INSTALL_DIR}/certs/ca/ca-cert.pem
TLSCertificateFile=${INSTALL_DIR}/certs/server/server-cert.pem
TLSCertificateKeyFile=${INSTALL_DIR}/certs/server/server-key.pem
TLSCertificateKeyFilePasswd=DqOUlNa1c2OMcEEbvTG30w==
TLSVerifyCert=never

[gbase_ssl_client]
TLSCACertificateFile=${INSTALL_DIR}/certs/ca/ca-cert.pem
TLSCertificateFile=${INSTALL_DIR}/certs/dba1/dba1-cert.pem
TLSCertificateKeyFile=${INSTALL_DIR}/certs/dba1/dba1-key.pem
TLSCertificateKeyFilePasswd=DqOUlNa1c2OMcEEbvTG30w==
TLSVerifyCert=never
!
chown ${USER_NAME}:${USER_NAME} ${INSTALL_DIR}/etc/sqlhosts.ext 2>/dev/null

####### 
loginfo "Set audit on."
sed -i "s#^ADTMODE.*#ADTMODE 1#g" ${INSTALL_DIR}/aaodir/adtcfg
sed -i "s#/usr/gbasedbt/aaodir#/home/dbaao/aaodir#g" ${INSTALL_DIR}/aaodir/adtcfg

####### restart database
loginfo "Restart GBase 8s Database Server."
su - ${USER_NAME} -c "timeout 1800 onmode -ky"
sleep 5
su - ${USER_NAME} -c "oninit -vy"
if [ $? -eq 0 ]; then
  su - dbsso -c "onaudit -a -u _default -e +DRDB,DRTB,ALTB,TCTB,STSN" 2>/dev/null
  loginfo "Finish."
else
  loginfo "Restart Server with ERROR. Please reduce the memory parameters(SHMVIRTSIZE,BUFFPOOLS), and then restart the server."
fi
cat <<EOF

--== GBase 8s Information for this install ==--
\$${USER_UPPER}SERVER : $GBASESERVER
\$${USER_UPPER}DIR    : $INSTALL_DIR
USER HOME       : $USER_HOME
DBSPACE DIR     : $DATADIR
IP ADDRESS      : ${IPADDR:-0.0.0.0}
PORT NUMBER     : $PORTNO
\$DB_LOCALE      : $GBASELOCALE
\$CLIENT_LOCALE  : $GBASELOCALE
JDBC URL        : jdbc:${USER_NAME}-sqli://IPADDR:$PORTNO/testdb:${USER_UPPER}SERVER=$GBASESERVER;DB_LOCALE=$GBASELOCALE;CLIENT_LOCALE=$GBASELOCALE;IFX_LOCK_MODE_WAIT=10
JDBC USERNAME   : $USER_NAME
JDBC PASSWORD   : $USER_PASS

WARNING!!! CANNOT be used in a production environment !!!
WARNING!!! Only for a test environment or development environment.

EOF

exit 0
