#!/bin/bash
# filename: backup.sh
# function: GBase 8s Database Server backup shell.
# write by: liaojinqing@gbase.cn 2022-01-25
export LANG=C
loginfo(){
  echo -e "[$(date +'%Y-%m-%d %H:%M:%S')] $*"
}

## define env
# USERNAME: gbasedbt OR informix
# 指定备份方式及日期，L0DAY表示0级备份日期1，4表示周1和周4，8/9表示无效日期
L0DAY=1,4
L1DAY=0,2,3,5,6
L2DAY=8
KEEPDAYS=7
WORKDIR=$(pwd)
USERNAME=gbasedbt
DBSTATUS=$(onstat -c|grep 'On-Line'|wc -l)
BACKUPDIR=$(onstat -c TAPEDEV)
LOGSDIR=$(onstat -c LTAPEDEV)
CURRDAY=$(date +%w)
HADBACKUP=$(onstat -g arc | awk '$1=="name"{need=1}need==1&&$3=="0"{print "1";exit}')

## check env
if [ ! x"$(whoami)" = "x${USERNAME}" ]; then
  loginfo "Must run as user: ${USERNAME}"
  exit 100
fi 

if [ ! "x${DBSTATUS}" = "x1" ]; then
  loginfo "Backup ONLY run On-Line Server which is Primary Server or Standard Server."
  exit 4
fi

if [ ! -d ${BACKUPDIR} ];then
  loginfo "The Backup dir is not exists."
  exit 3
fi

if [ ! -d ${LOGSDIR} ];then
  loginfo "The logs backup dir is not exists."
  exit 2
fi

## backup db
# delete file
loginfo "Delete files at $BACKUPDIR which out ${KEEPDAYS:-30} days."
cd $BACKUPDIR && find $BACKUPDIR -name "*L[0-2]" -type f -ctime +${KEEPDAYS:-30} | xargs rm -rf
loginfo "Delete files at $LOGSDIR which out ${KEEPDAYS:-30} days."
cd $LOGSDIR && find $LOGSDIR -name "*0_Log*" -type f -ctime +${KEEPDAYS:-30} | xargs rm -rf
# backup db
if [ x"$HADBACKUP" = x ]; then
  loginfo "Execute L0 backup start, while no backup found."
  CURRDAY=9
  ontape -s -L 0
fi

if [[ ${L0DAY} =~ ${CURRDAY} ]]; then
  loginfo "Execute L0 backup start..."
  ontape -s -L 0
elif [[ ${L1DAY} =~ ${CURRDAY} ]]; then
  loginfo "Execute L1 backup start..."
  ontape -s -L 1
elif [[ ${L2DAY} =~ ${CURRDAY} ]]; then
  loginfo "Execute L2 backup start..."
  ontape -s -L 2
fi

if [ $? -eq 0 ];then
  loginfo "Execute backup finished."
  exit 0
else
  loginfo "Execute backup error."
  exit 1
fi

