#!/bin/bash
# filename: backup_gz.sh
# function: GBase 8s Database Server backup shell.
# crontab(root)    : 0 1 * * * su - gbasedbt -c "/opt/gbase/script/backup_gz.sh" >/dev/null 2>&1
# crontab(gbasedbt): 0 1 * * * . /home/gbase/.bash_profile && /opt/gbase/script/backup_gz.sh >/dev/null 2>&1
# write by: liaosnet@gbasedbt.com 2024-04-25
export LANG=C
WORKDIR=$(cd $(dirname $0) && pwd)
LOGFILE=${WORKDIR}/status_backup_gz.log
[ ! -f ${LOGFILE} ] && touch ${LOGFILE}
loginfo(){
  echo -e "[$(date +'%Y-%m-%d %H:%M:%S')] $*" >> ${LOGFILE}
}

## define env
# USERNAME: gbasedbt OR informix
# L0DAY: level 0 backup
# L1DAY: level 1 backup
# L2DAY: level 2 backup
# 0 = Sun ... 6 = Sat, other = Invalid Date
L0DAY=6
L1DAY=0,1,2,3,4,5
L2DAY=8
KEEPDAYS=14
USERNAME=gbasedbt
DBSTATUS=$(onstat -c|grep 'On-Line'|wc -l)
BACKUPDIR=$(onstat -c TAPEDEV)
LOGSDIR=$(onstat -c LTAPEDEV)
LOGNEED=1
CURRDAY=$(date +%w)
HADBACKUP=$(onstat -g arc | awk '$1=="name"{need=1}need==1&&$3=="0"{print "1";exit}')

## check env
if [ ! x"$(whoami)" = "x${USERNAME}" ]; then
  loginfo "Must run as user: ${USERNAME}"
  exit 100
fi 

if [ ! "x${DBSTATUS}" = "x1" ]; then
  loginfo "Backup ONLY run On-Line Server which is Primary Server or Standard Server."
  exit 4
fi

if [ ! -d ${BACKUPDIR} ]; then
  loginfo "The Backup dir is not exists."
  exit 3
fi

[ x"${LOGSDIR}" = "x/dev/null" ] && LOGNEED=0
if [[ ${LOGNEED} -eq 1 && ! -d "${LOGSDIR}" ]]; then
  loginfo "The logs backup dir is not exists."
  LOGNEED=0
fi

ZIPTYPE=gz
ZIPCMD=gzip
if [ -x /usr/bin/xz ]; then
  ZIPTYPE=xz
  ZIPCMD=xz
fi

## backup db
# delete file
KEEPDAYS=${KEEPDAYS:-30}
loginfo "Delete files at $BACKUPDIR which out ${KEEPDAYS} days."
# function find had different action for path。
# or find . -name "*L[0-2]" -type f -mtime +${KEEPDAYS} -exec rm -rf {} \;
cd $BACKUPDIR && find . -name "*L[0-2].[gx]z" -type f -mtime +${KEEPDAYS} | xargs rm -rf
if [ ${LOGNEED} -eq 1 ]; then
  loginfo "Delete files at $LOGSDIR which out ${KEEPDAYS} days."
  cd $LOGSDIR && find . -name "*_Log*.[gx]z" -type f -mtime +${KEEPDAYS} | xargs rm -rf
fi

if [[ ${L0DAY} =~ ${CURRDAY} ]]; then
  loginfo "Execute L0 backup start..."
  ontape -s -L 0 >> ${LOGFILE} 2>&1
elif [[ ${L1DAY} =~ ${CURRDAY} ]]; then
  loginfo "Execute L1 backup start..."
  ontape -s -L 1 >> ${LOGFILE} 2>&1
elif [[ ${L2DAY} =~ ${CURRDAY} ]]; then
  loginfo "Execute L2 backup start..."
  ontape -s -L 2 >> ${LOGFILE} 2>&1
fi

BAKSTATUS=$?
if [ ${BAKSTATUS} -eq 0 ];then
  loginfo "Execute backup finished."
else
  loginfo "Execute backup error! Try Execute L0 backup start."
  ontape -s -L 0 >> ${LOGFILE} 2>&1
  BAKSTATUS=$?
  loginfo "Execute backup finished with CODE: ${BAKSTATUS}"
fi

loginfo "Change filename for data backup if need."
cd $BACKUPDIR
for FILE in $(ls *L[0-2] 2>/dev/null)
do
  # modify ${HOSTNAME}_${SERVERNUM}_L${LEVEL} to ${HOSTNAME}_${SERVERNUM}_yyyymmdd_hh24miss_L${LEVEL}
  FILETIME=$(stat $FILE | awk '/Modify:/{gsub("-","");gsub(":","");print $2"_"substr($3,1,6)}')
  NEWFILE=${FILE%_*}_${FILETIME}_${FILE##*_}
  SERVNUM=${FILE%_*}
  SERVNUM=${SERVNUM##*_}
  if [ ! ${#SERVNUM} -eq 6 ]; then
    loginfo "Rename ${FILE} to ${NEWFILE}"
    mv ${FILE} ${NEWFILE}
  fi
done
cd $BACKUPDIR
DATAFILE=$(find . -name "*L[0-2]" -type f)
if [ ! x"${DATAFILE}" = x ];then
  loginfo "${ZIPCMD} data backup."
  loginfo "${ZIPCMD} file(s): ${DATAFILE}"
  ${ZIPCMD} ${DATAFILE}
fi
if [ ${LOGNEED} -eq 1 ]; then
  cd $LOGSDIR 
  LOGSFILE=$(find . -name "*_Log*" ! -name "*.[gx]z" -type f)
  if [ ! x"${LOGSFILE}" = x ];then
    loginfo "${ZIPCMD} logs backup."
    loginfo "${ZIPCMD} file(s): ${LOGSFILE}"
    ${ZIPCMD} ${LOGSFILE}
  fi
fi

exit 0
