#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "gci.h"

#define BUFFER_LEN      1000
#define ERRMSG_LEN      200
#define ROWS_EACH_FETCH	5
#define GCI_IGNORE      0

#define SQLMSG \
	sb1 errmsg[256] = {0}, errstate[256] = {0}; \
	sb4 errcode; \
	GCIErrorGet(errhp, 0, errstate, &errcode, errmsg, 256, GCI_HTYPE_ERROR); \
	printf("Error code ----- %d\nError state ---- %s\nError message -- %s\n", errcode, errstate, errmsg)

typedef struct {
	sb1 *buf;
	ub4 pos; 
	ub4 unit;
	ub4 sql_t;
	ub4 pwid;
} Col;

int rc = GCI_SUCCESS;
GCIEnv *envhp = NULL;
GCISvcCtx *svchp = NULL;
GCIError *errhp = NULL;
GCIStmt *stmthp = NULL;
GCIDefine *bndhp[20];
GCILobLocator *lobs[ROWS_EACH_FETCH];  // assume one lob column
ub4 isUdoEnabled;
char *colname = {0};

char clob_buf[BUFFER_LEN];

#define $GCIDefineByPos(i, pos, buf, unit, type) \
rc = GCIDefineByPos( \
	stmthp,      \
	&bndhp[i],   \
	errhp,       \
	pos,         \
	buf,         \
	unit,        \
	type,        \
	0,           \
	0,           \
	0,           \
	GCI_DEFAULT  \
); \
do { if (rc != GCI_SUCCESS) { SQLMSG; error("GCIDefineByPos\n"); } } while (0)

#define $GCIDefineArrayOfStruct(i, rowsz) \
rc = GCIDefineArrayOfStruct(bndhp[i], errhp, rowsz, 0, 0, 0); \
do { if (rc != GCI_SUCCESS) { SQLMSG; error("GCIDefineArrayOfStruct\n"); } } while (0)

#define $GCIDescriptorAlloc(i) \
rc = GCIDescriptorAlloc(     \
	envhp,               \
	(dvoid **)&lobs[i],  \
	GCI_DTYPE_LOB,       \
	0,                   \
	NULL                 \
);                           \
do { if (rc != GCI_SUCCESS) { SQLMSG; error("GCIDescriptorAlloc\n"); } } while (0)

void cleanup()
{
/* CLEANUP: Close the statement handle
**          Free the statement handle
**          Disconnect from the datasource
**          Free the connection and environment handles
**          Exit
*/

	GCILogoff(svchp, errhp);
	if (svchp)
		GCIHandleFree((dvoid *)svchp, GCI_HTYPE_SVCCTX);
	if (errhp)
		GCIHandleFree((dvoid *)errhp, GCI_HTYPE_ERROR);
	if (envhp)
		GCIHandleFree((dvoid *)envhp, GCI_HTYPE_ENV);
	printf("\n\nHit <Enter> to terminate the program...\n\n");
	getchar();
}

void error(const char *msg)
{
	fprintf(stdout, "ERROR: ");
	fprintf(stdout, msg);
}

void die(const char *msg)
{
	fprintf(stdout, msg);
	cleanup();
	exit(rc);
}


char *trim(char *s)
{
	char *p;
	for (p = s; *p == '\n' || *p == '\r';) p++;
	return p;
}

sb1* read_batch(const char *path)
{
	FILE *fp;
	long size;
	char *s, *token;

	fp = fopen(path, "r");
	fseek(fp, 0, SEEK_END);
	size = ftell(fp);
	if (size == 0) {
		printf("Empty file: %s\n", path);
		goto error;
	} else if (s = calloc(1, size)) {
		rewind(fp);
		while (!feof(fp)) {
			fread(s, size, 1, fp);
			if (ferror(fp)) {
				printf("Error reading file: %s\n", path);
				free(s);
				goto error;
			}
		}
		fclose(fp);
		return s;
	} else {
		printf("malloc failed for %s\n", path);
		goto error;
	}

error:
	fclose(fp);
	return 0;
}

int exec_sql(const char *sql)
{
	int rc;
	printf("SQL> %s\n", sql);
	rc = GCIStmtExecDirect(
		svchp,
		sql,
		errhp,
		0,
		0,
		0,
		0,
		GCI_DEFAULT
	);
	return rc;
}

int exec_batch(const char *path)
{
	char *str, *token;
	int rc;

	str = read_batch(path);
	if (str == 0) {
		printf("Got nothing from %s\n", path);
		return -1;
	}

	token = trim(strtok(str, ";"));
	while (token != 0 && strlen(token)) {
		rc = exec_sql(token);
		if (rc != GCI_SUCCESS) {
			SQLMSG;
		}
		token = trim(strtok(NULL, ";"));
	}
	free(str);
	return 0;
}

void init_env_handle()
{
	/* initialize to threading mode and object environment */
	rc = GCIEnvCreate(
		&envhp,
		GCI_DEFAULT|GCI_OBJECT,
		(dvoid *)0,
		0,
		0,
		0,
		(size_t) 0,
		(dvoid **)0
	);

	if (rc != GCI_SUCCESS) {
		die(
			"Error in Step 1 -- GCIEnvCreate: "
			"create env handle failed\nExiting!!\n"
		);
	}
}

void init_error_handle()
{
	rc = GCIHandleAlloc(
		(dvoid *)envhp,
		(dvoid **)&errhp,
		GCI_HTYPE_ERROR,
		0,
		(dvoid **)0
	);

	if (rc != GCI_SUCCESS) {
		die(
			"Error in Step 1 -- GCIHandleAlloc: "
			"allocate error handle failed\nExiting!!\n"
		);
	}
}

void init_svctx_handle()
{
	rc = GCIHandleAlloc(
		(dvoid *)envhp,
		(dvoid **)&svchp,
		GCI_HTYPE_SVCCTX,
		0,
		(dvoid **)0
	);

	if (rc != GCI_SUCCESS) {
		die(
			"Error in Step 1 -- GCIHandleAlloc: "
			"allocate svcctx handle failed\nExiting!!\n"
		);
	}
}


void logon(const char *user,
           const char *pswd,
           const char *dbname)
{
	rc = GCILogonNoDb(
		envhp,
		errhp,
		&svchp,
	       	user,
		(ub4)strlen((char*)user),
		pswd,
		(ub4)strlen((char*)pswd),
		dbname,
		(ub4)strlen((char*)dbname)
	);

	if (rc != GCI_SUCCESS) {
		SQLMSG;
		die(
			"Error in Step 1 -- GCILogon: "
			"logon to database failed\nExiting!!\n"
		);
	}
}

void check_server_version()
{
	ub1 verInfoBuffer[BUFFER_LEN];
	ub1 majorVer[3];

/*  Get version information from the database server.
 *  Set the value of isUdoEnabled depending on the database
 *  version returned
 */
	rc = GCIServerVersion(
		svchp,
		errhp,
		verInfoBuffer,
		BUFFER_LEN,
		GCI_DEFAULT
	);

	if (rc != GCI_SUCCESS) {
		die(
			"Error in Step 2 -- GCIServerVersion: "
			"get database server version failed\nExiting!!\n"
		);
	}

	/* Set the value of isUDOEnabled */
	strncpy((char *)majorVer, (char *)verInfoBuffer, 2);
	if (strncmp(majorVer, "09", 2) >= 0 )
		isUdoEnabled = 1;
	else
		isUdoEnabled = 0;

	/* Use no udo, GCIBindByPos will not concern of INPUT/OUTPUT type of ODBC */
	isUdoEnabled = 0;
	printf("Database Version -- %s\n", verInfoBuffer);
}

void create_database()
{
	if (isUdoEnabled) {
		printf("Creating UDO database\n");
		rc = exec_batch("./dbcreate.sql");
	} else {
		printf("Creating non-UDO database\n");
		rc = exec_batch("./dbcreate.sql");
	}

	if (rc != 0) {
		die("");
	}
	printf("Sample Database Created\n");
}

void import_data()
{
	printf("Populating table 'customer'\n");
	rc = exec_batch("./customer_data.sql");
	if (rc != 0) {
		die("");
	}

	printf("Populating table 'item'\n");
	rc = exec_batch("./item_data.sql");
	if (rc != 0) {
		die("");
	}

	printf("Populating table 'orders'\n");
	rc = exec_batch("./orders_data.sql");
	if (rc != 0) {
		die("");
	}
}


void init_stmt_handle()
{
	rc = GCIHandleAlloc(
		(dvoid *)envhp,
		(dvoid **)&stmthp,
		GCI_HTYPE_STMT,
		0,
		0
	);

	if (rc != GCI_SUCCESS) {
		die("");
	}	
}


int main(int argc, char *argv[])
{
	GCItext *dbname = (GCItext*)"testgci";//gci_demodb
	GCItext *user = (GCItext*)"gbasedbt";
	GCItext *pswd = (GCItext*)"Big4ifmx";

	init_env_handle();
	init_error_handle();
	init_svctx_handle();

	logon(user, pswd, dbname);
	check_server_version();

	create_database();
	import_data();

	Col cs1[] = {
		{NULL, 1, 10,  SQLT_STR, 10},
		{NULL, 2,  4,  SQLT_INT, 10},
	};
	rc = do_select_storebycol(
		"SELECT fname, cust_num FROM customer WHERE cust_num=101",
		cs1,
		sizeof(cs1)/sizeof(Col)
	);
	if (rc != GCI_SUCCESS) {
		die("");
	}

	rc = exec_sql("UPDATE customer SET fname='JUly' WHERE cust_num=101");
	if (rc != GCI_SUCCESS) {
		die("");
	}

	rc = do_select_storebycol(
		"SELECT fname, cust_num FROM customer WHERE cust_num=101",
		cs1,
		sizeof(cs1)/sizeof(Col)
	);
	if (rc != GCI_SUCCESS) {
		die("");
	}

	Col cs2[] = {
		{NULL, 1,  4,  SQLT_INT, 10},
		{NULL, 2,  4,  SQLT_INT, 10},
	};
	rc = do_select_storebycol(
		"SELECT quantity, item_num FROM orders WHERE item_num=1001",
		cs2,
		sizeof(cs2)/sizeof(Col)
	);
	if (rc != GCI_SUCCESS) {
		die("");
	}

	rc = exec_sql("UPDATE orders SET quantity=888 WHERE item_num=1001");
	if (rc != GCI_SUCCESS) {
		die("");
	}

	rc = do_select_storebycol(
		"SELECT quantity, item_num FROM orders WHERE item_num=1001",
		cs2,
		sizeof(cs2)/sizeof(Col)
	);
	if (rc != GCI_SUCCESS) {
		die("");
	}

	Col cs3[] = {
		{NULL, 1, 10,  SQLT_STR, 10},
	};
	rc = do_select_storebycol(
		"SELECT fname FROM customer WHERE lname='Miller'",
		cs3,
		sizeof(cs3)/sizeof(Col)
	);
	if (rc != GCI_SUCCESS) {
		die("");
	}

	Col cs4[] = {
		{NULL, 1,  4,  SQLT_INT, 10},
		{NULL, 2, sizeof(GCIDate),  SQLT_DAT, 16},
		{NULL, 3,  4,  SQLT_INT, 10},
	};
	rc = do_select_storebycol(
		"SELECT cust_num, order_date, quantity FROM orders WHERE item_num=1001",
		cs4,
		sizeof(cs4)/sizeof(Col)
	);
	if (rc != GCI_SUCCESS) {
		die("");
	}

	Col cs5[] = {
		{NULL, 1,  4,  SQLT_INT, 10},
		{NULL, 2, 10,  SQLT_STR, 10},
		{NULL, 3, 10,  SQLT_STR, 10},
	};
	rc = do_select_storebycol(
		"SELECT cust_num, fname, lname FROM customer WHERE lname='Miller'",
		cs5,
		sizeof(cs5)/sizeof(Col)
	);
	if (rc != GCI_SUCCESS) {
		die("");
	}

/*
	Col cs6[] = {
		{NULL, 1,  4,  SQLT_INT, 10},
		{NULL, 2, 20,  SQLT_STR, 20},
		{NULL, 3,  4,  SQLT_INT,  6},
		{NULL, 4, 10,  SQLT_STR, 10},
		{NULL, 5,  8,  SQLT_STR, 11},
		{NULL, 6,  4,  SQLT_CLOB, 20},
	};                     // unit_price STR ?
	rc = do_select_storebycol(
		"SELECT * FROM item",
		cs6,
		sizeof(cs6)/sizeof(Col)
	);
	if (rc != GCI_SUCCESS) {
		die("");
	}
*/

	Col cs6[] = {
		{NULL, 1,  4,  SQLT_CLOB, 20},
	};
	rc = do_select_storebycol(
		"SELECT advert FROM item",
		cs6,
		sizeof(cs6)/sizeof(Col)
	);
	if (rc != GCI_SUCCESS) {
		SQLMSG;
	}

	Col cs7[] = {
		{NULL, 1,  4,  SQLT_INT, 10},
	};
	rc = do_select_storebycol(
		"SELECT aaa FROM orders WHERE item_num=1001",
		cs7,
		sizeof(cs7)/sizeof(Col)
	);
	if (rc != GCI_SUCCESS) {
		SQLMSG;
	}

	rc = exec_sql("DELETE FROM orders WHERE item_num=1004");
	if (rc != GCI_SUCCESS) {
		error("");
	}

	rc = exec_batch("./dbdrop.sql");
	if (rc != 0) {
		error("");
	}

	cleanup();
}

sb4 get_cols()
{
	sb4 cols;
	rc = GCIAttrGet(
		stmthp,
		GCI_HTYPE_STMT,
		&cols,
		0,
		GCI_ATTR_NUM_COLS,
		errhp
	);

	if (rc != GCI_SUCCESS) {
                SQLMSG;
		error("GCIAttrGet for GCI_ATTR_NUM_COLS");
		return -1;
	}
	//printf("Total of cols: %d\n", cols);
	return cols;
}

int get_col_name(int i)
{
	rc = GCIAttrGet(
		stmthp,
		GCI_HTYPE_STMT,
		colname,
		0,
		GCI_ATTR_NAME,
		errhp
	);

	if (rc != GCI_SUCCESS) {
		error("GCIAttrGet for GCI_ATTR_NAME\n");
		return rc;
	}

	GCIParam *param = NULL;
	rc = GCIParamGet(
		(CONST dvoid *)stmthp,
		(ub4)GCI_HTYPE_STMT,
		(GCIError *)errhp,
		(dvoid **)&param,
		(ub4)i
	);

	if (rc != GCI_SUCCESS) {
		error("GCIParamGet\n");
		return rc;
	}

	rc = GCIAttrGet(
		param,
		GCI_DTYPE_PARAM,
		&colname,
		0,
		GCI_ATTR_NAME,
		errhp
	);

	if (rc != GCI_SUCCESS) {
		error("GCIAttrGet for GCI_ATTR_NAME\n");
		return rc;
	}
}

void prepare_col_buffer(Col cols[], int n)
{
	int j;
	for (j = 0; j < n; j++) {
		switch (cols[j].sql_t) {
		case SQLT_CLOB:
			$GCIDescriptorAlloc(0);
			$GCIDescriptorAlloc(1);
			$GCIDescriptorAlloc(2);
			$GCIDescriptorAlloc(3);
			$GCIDescriptorAlloc(4);

			$GCIDefineByPos(
				j,
				cols[j].pos,
				lobs,
				cols[j].unit,
				cols[j].sql_t
			);
			break;
		default:
			cols[j].buf = (sb1 *)calloc(
				ROWS_EACH_FETCH, cols[j].unit
			);
			if (!cols[j].buf) {
				die("malloc col buf failed!");
			}
			$GCIDefineByPos(
				j,
				cols[j].pos,
				cols[j].buf,
				cols[j].unit,
				cols[j].sql_t
			);
		}
	}
}

void free_col_buffer(Col cols[], int n)
{
	int j;
	for (j = 0; j < n; j++) {
		switch (cols[j].sql_t) {
		case SQLT_CLOB:
			GCIDescriptorFree(lobs[0], GCI_DTYPE_LOB);
			GCIDescriptorFree(lobs[1], GCI_DTYPE_LOB);
			GCIDescriptorFree(lobs[2], GCI_DTYPE_LOB);
			GCIDescriptorFree(lobs[3], GCI_DTYPE_LOB);
			GCIDescriptorFree(lobs[4], GCI_DTYPE_LOB);
			lobs[0] = NULL;
			lobs[1] = NULL;
			lobs[2] = NULL;
			lobs[3] = NULL;
			lobs[4] = NULL;
			break;
		default:
			free(cols[j].buf);
		}
		bndhp[j] = NULL;
	}
}

int nrows_fetched()
{
	int rc, rows = 0;
	rc = GCIAttrGet(
		stmthp,
		GCI_HTYPE_STMT,
		&rows,
		0,
		GCI_ATTR_ROW_COUNT,
		errhp
	);

        if (rc != GCI_SUCCESS) {
		printf("Fetched %d rows!\n", rows);
		return rc;
	}
	return rows;
}

int get_lob_size(GCILobLocator *lob)
{
	int rc;
	ub4 sz;
	rc = GCILobGetLength(
		svchp, errhp, lob, &sz
	);
	if(rc != GCI_SUCCESS) {
		printf("error getting lob size: %d\n", sz);
		SQLMSG;
		return rc;
	}
	return sz;
}

int present_clob(GCILobLocator *lob)
{
	int size, read;
	size = get_lob_size(lob);

	while (size) {
		if (size > BUFFER_LEN) {
			read = BUFFER_LEN;
		} else {
			read = size;
		}

		rc =  GCILobRead(
			svchp,
			errhp,
			lob,
			&read,
			GCI_IGNORE,
			clob_buf,
			read,
			NULL,
			NULL,
			0,
			0
		);

		if (rc != GCI_SUCCESS && rc != GCI_SUCCESS_WITH_INFO) {
			error("GCILobRead\n");
			return rc;
		}
		printf("%s", clob_buf);
		size -= read;
	}
	return 0;
}

int present_row(int i, Col cols[], int n)
{
	int j, rc = 0;
	for (j = 0; j < n; j++) {
		switch (cols[j].sql_t) {
		case SQLT_INT:
			printf("%*d", cols[j].pwid,
				*((int*)cols[j].buf+i));
			break;
		case SQLT_STR:
			printf("%*s", cols[j].pwid,
				cols[j].buf+i*cols[j].unit);
			break;
		case SQLT_DAT:
			printf("%*c%04d-%02d-%02d",
				cols[j].pwid - 10,
				' ',
				((GCIDate*)cols[j].buf+i)->GCIDateYYYY,
				((GCIDate*)cols[j].buf+i)->GCIDateMM,
				((GCIDate*)cols[j].buf+i)->GCIDateDD);
			break;
		case SQLT_FLT:
			printf("%*.2f", cols[j].pwid,
				*((float*)cols[j].buf+i));
			break;
		case SQLT_CLOB:
			rc = present_clob(lobs[i]);
		}
	}
	return rc;
}

int do_select_storebycol(GCItext *sql, Col cols[], int n)
{
	int i, j;
	printf("SQL> %s\n\n", sql);
	init_stmt_handle();
	rc = GCIStmtPrepare(
		stmthp,
		errhp,
		sql,
		strlen(sql),
		0,
		GCI_DEFAULT
	);
	if (rc != GCI_SUCCESS) {
		error("GCIStmtPrepare\n");
		return rc;
	}

	prepare_col_buffer(cols, n);

	/* Execute prepared statement */
	rc = GCIStmtExecute(
		svchp,
		stmthp,
		errhp,
		GCI_IGNORE,
		GCI_IGNORE,
		GCI_IGNORE,
		GCI_IGNORE,
		GCI_COMMIT_ON_SUCCESS
	);
	if (rc != GCI_SUCCESS) {
		SQLMSG;
		error("GCIStmtExecute\n");
		return rc;
	}

	sb4 ncols = get_cols();
	printf("%5c", ' ');
	for (i = 0; i < ncols; i++) {
		get_col_name(i+1);
		printf("%*s", cols[i].pwid, colname);
	}
	printf("\n");

	int prows = 0;
	while (1) {
		int rows;
		rc = GCIStmtFetch(
			stmthp,
			errhp,
			ROWS_EACH_FETCH,
			GCI_FETCH_NEXT,
			GCI_DEFAULT
		);

		if (rc == GCI_NO_DATA) {
			rc = GCI_SUCCESS;
			goto SUCCESS_EXIT;
		}

		rows = nrows_fetched();
		for (i=0; i<rows-prows; i++) {
			printf("\n%4d:", prows+1+i);
			present_row(i, cols, n);
		}
		prows = rows;
	}

SUCCESS_EXIT:
	printf("\n\n");
	free_col_buffer(cols, n);
	GCIHandleFree((dvoid *)stmthp, GCI_HTYPE_STMT);
	return rc;
}
