/***************************************************************************
*	Licensed Materials - Property of GBase
*
*
*	"Restricted Materials of GBase"
*
*
*
*	GBase 8s GCI Interface demo
*
*
*	Copyright GBase Corporation 2015 All rights reserved.
*
*
*
*
*
*  Title:          test_dbcreate.c
*
*  Description:    To create & populate the sample database. The schema
*                  for the sample database is as follows -
*
*  For servers supporting UDO (version no. > 9)
*
*	Database - odbc_demodb (with log)
*
*      Table - CUSTOMER
*          cust_num	      INTEGER, PRIMARY KEY
*          fname   		  VARCHAR(10)
*          lname	      VARCHAR(10)
*          address		  ROW (address1	VARCHAR(20),
*			                   city		VARCHAR(15),
*			                   state	VARCHAR(5),
*			                   zip		VARCHAR(6)
*                             )
*          contact_dates  LIST (DATETIME YEAR TO DAY NOT NULL)
*
*
*      Table - ITEM
*	       item_num	      INTEGER, PRIMARY KEY
*          description	  VARCHAR(20)
*	       stock		  SMALLINT
*	       ship_unit	  VARCHAR(10)
*	       advert		  CLOB
*	       unit_price	  DISTINCT TYPE dollar
*
*
*      Table - ORDERS
*	       order_num	  INTEGER, PRIMARY KEY
*	       cust_num	      INTEGER, FOREIGN KEY, REFERENCES customer(customer_num)
*          item_num	      INTEGER, FOREIGN KEY, REFERENCES item(item_num)
*          order_date	  DATE
*          quantity		  INTEGER
*
*
*
*  For servers not supporting UDO (version no. < 9)
*
*	Database - odbc_demodb (with log)
*
*      Table - CUSTOMER
*          cust_num	      INTEGER, PRIMARY KEY
*          fname		  VARCHAR(10)
*          lname		  VARCHAR(10)
*          address1       VARCHAR(20)
*          city           VARCHAR(15)
*          state          VARCHAR(5)
*          zip            VARCHAR(6)
*          first_contact  DATETIME YEAR TO DAY
*          last_contact   DATETIME YEAR TO DAY
*
*
*      Table - ITEM
*          item_num	      INTEGER, PRIMARY KEY
*      	   description	  VARCHAR(20)
*          stock		  SMALLINT
*      	   ship_unit	  VARCHAR(10)
*          advert         VARCHAR(100)
*      	   unit_price	  DECIMAL
*
*
*      Table - ORDERS
*      	   order_num	  INTEGER, PRIMARY KEY
*      	   cust_num	      INTEGER, FOREIGN KEY, REFERENCES customer(customer_num)
*          item_num	      INTEGER, FOREIGN KEY, REFERENCES item(item_num)
*          order_date	  DATE
*          quantity		  INTEGER
*
*
***************************************************************************
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "gci.h"
#include "test_dbcreate.h"

#define ERRMSG_LEN      200

#define SQLMSG 			sb1 errmsg[256] = {0}, errstate[256] = {0};\
		sb4 errcode;\
		GCIErrorGet(errhp, 0, errstate, &errcode, errmsg, 256, GCI_HTYPE_ERROR);\
		fprintf(stdout, "Error code ----- %d\nError state ---- %s\nError message -- %s\n", errcode, errstate, errmsg);	

int main (long         argc,
          char*        argv[])
{
    /* Declare variables */    
    ub4			i, in;

    ub1         verInfoBuffer[BUFFER_LEN];
    ub4			verInfoLen;
    ub1         majorVer[3];
    ub4      	isUdoEnabled;

    ub1         insertStmt[BUFFER_LEN];

    /* Handles */
	GCIEnv *envhp = NULL;
	GCISvcCtx *svchp = NULL;
	GCIError *errhp = NULL;

	int rc = GCI_SUCCESS;

	GCItext *dbname = (GCItext*)"gci_demodb";//"gci_demodb";
	GCItext *user = (GCItext*)"root";

	GCItext *pswd = (GCItext*)"111111";

    /*  STEP 1. Get data source name from command line (or use default)
    **          Allocate the environment handle and set ODBC version
    **          Allocate the connection handle
    **          Establish the database connection
    **          Allocate the statement handle
    **          Drop demo database if it already exists
    */

	/* initialize the mode to be the threaded and object environment */
	if(rc = GCIEnvCreate(&envhp, GCI_THREADED|GCI_OBJECT, (dvoid *)0,
			0, 0, 0, (size_t) 0, (dvoid **)0) != GCI_SUCCESS)
	{
		fprintf(stdout, "Error in Step 1 -- GCIEnvCreate: create env handle failed\nExiting!!\n");
		goto EXIT;
	}
	
	/* allocate an error handle */
	if(rc = GCIHandleAlloc((dvoid *)envhp, (dvoid **)&errhp,
			GCI_HTYPE_ERROR, 0, (dvoid **) 0) != GCI_SUCCESS)
	{
		fprintf(stdout, "Error in Step 1 -- GCIHandleAlloc: allocate error handle failed\nExiting!!\n");
		goto EXIT;
	}	
	
	if(rc = GCIHandleAlloc((dvoid *)envhp, (dvoid **)&svchp,
			GCI_HTYPE_SVCCTX, 0, (dvoid **) 0) != GCI_SUCCESS)
	{
		fprintf(stdout, "Error in Step 1 -- GCIHandleAlloc: allocate svcctx handle failed\nExiting!!\n");
		goto EXIT;
	}

	if(rc = GCILogonNoDb(envhp, errhp, &svchp, user, (ub4)strlen((char*)user),
			pswd, (ub4)strlen((char*)pswd), dbname, (ub4)strlen((char*)dbname)) != GCI_SUCCESS)
	{
	SQLMSG
		fprintf(stdout, "Error in Step 1 -- GCILogon: logon to database failed\nExiting!!\n");
		goto EXIT;
	}
	
	/* Drop demo database if it already exists */
    rc = GCIStmtExecDirect (svchp, (ub1*) "DROP DATABASE gci_demodb", errhp, 0, 0, 0, 0, GCI_DEFAULT);

	fprintf(stdout, "Step 1 done...Connected to database server success\n");
	


    /* STEP 2.  Get version information from the database server
    **          Set the value of isUdoEnabled depending on the database version returned
    */

    /* Get version information from the database server */
    if(rc = GCIServerVersion(svchp, errhp, verInfoBuffer, BUFFER_LEN, GCI_DEFAULT) != GCI_SUCCESS)
	{
		fprintf(stdout, "Error in Step 2 -- GCIServerVersion: get database server version failed\nExiting!!\n");
		goto EXIT;
	}    

    /* Set the value of isUDOEnabled */
    strncpy ((char *) majorVer, (char *) verInfoBuffer, 2);
    if (strncmp (majorVer, "09", 2) >= 0 )
        isUdoEnabled = 1;
    else
        isUdoEnabled = 0;
	
	/* Use no udo, GCIBindByPos will not concern of INPUT/OUTPUT type of ODBC */
	isUdoEnabled =0;
	
    fprintf (stdout, "Step 2 done...Database Version -- %s\n", verInfoBuffer);




    /* STEP 3.  Execute the SQL statements to create the database
    **          (depending on the version of the database server)
    */

    if (isUdoEnabled)
    {
        fprintf (stdout, "Creating UDO database\n");

        for (i = 0; i < NUM_DBCREATE_STMTS; i++)
        {
            fprintf (stdout, "Executing stmt %d of %d\n", i+1, NUM_DBCREATE_STMTS);

            if(rc = GCIStmtExecDirect (svchp, createUdoDbStmts[i], errhp, 0, 0, 0, 0, GCI_DEFAULT) != GCI_SUCCESS)
            {
                        	sb1 errmsg[256] = {0}, errstate[256] = {0};
				sb4 errcode;
				GCIErrorGet(errhp, 0, errstate, &errcode, errmsg, 256, GCI_HTYPE_ERROR);
				fprintf(stdout, "Error code ----- %d\nError state ---- %s\nError message -- %s\n", errcode, errstate, errmsg);
				
				fprintf(stdout, "Error in Step 3 -- GCIStmtExecDirect: direct execute statement failed\nExiting!!\n");
				
				goto EXIT;
			}  
        }
    }
    else
    {
        fprintf (stdout, "Creating non-UDO database\n");

        for (i = 0; i < NUM_DBCREATE_STMTS -1; i++)
        {
            fprintf (stdout, "Executing stmt %d\n", i+1, NUM_DBCREATE_STMTS -1);

            if(rc = GCIStmtExecDirect (svchp, createNonUdoDbStmts[i], errhp, 0, 0, 0, 0, GCI_DEFAULT) != GCI_SUCCESS)
            {
            	sb1 errmsg[256] = {0}, errstate[256] = {0};
				sb4 errcode;
				GCIErrorGet(errhp, 0, errstate, &errcode, errmsg, 256, GCI_HTYPE_ERROR);
				fprintf(stdout, "Error code ----- %d\nError state ---- %s\nError message -- %s\n", errcode, errstate, errmsg);
				
				fprintf(stdout, "Error in Step 3 -- GCIStmtExecDirect: direct execute statement failed\nExiting!!\n");
				goto EXIT;
			}  
        }
    }


    fprintf (stdout, "Step 3 done...Sample Database Created\n");




    /* STEP 4.  Construct the INSERT statement for table 'customer'
    **          Insert data into the table 'customer'. The data inserted depends on
    **          whether the database created is UDO or non-UDO enabled
    */

    fprintf (stdout, "Populating table 'customer'\n");

    for (i = 0; i < NUM_CUSTOMERS; i++)
    {
        fprintf (stdout, "Inserting row # %d\n", i+1);
        
        memset(insertStmt, 0, BUFFER_LEN);

        /* Construct the INSERT statement for table 'customer' */
        strcpy ((char *) insertStmt, (char *) insertDBStmts[0]);

        if (isUdoEnabled)
            strcat ((char *) insertStmt, (char *) udoCust[i]);
        else
            strcat ((char *) insertStmt, (char *) nonUdoCust[i]);

        /* Execute the INSERT statement */
        if(rc = GCIStmtExecDirect (svchp, insertStmt, errhp, 0, 0, 0, 0, GCI_DEFAULT) != GCI_SUCCESS)
        {
			fprintf(stdout, "Error in Step 4 -- GCIStmtExecDirect: direct execute statement failed\nExiting!!\n");
			goto EXIT;
		}  
    }


    fprintf (stdout, "Step 4 done...Data inserted into the 'customer' table\n");




    /* STEP 5.  Construct the INSERT statement for table 'item'
    **          Insert data into the table 'item'. The data inserted depends on
    **          whether the database created is UDO or non-UDO enabled
    */

    fprintf (stdout, "Populating table 'item'\n");

    for (i = 0; i < NUM_ITEMS; i++)
    {
        fprintf (stdout, "Inserting row # %d\n", i+1);
        
        memset(insertStmt, 0, BUFFER_LEN);

        /* Construct the INSERT statement for table 'item' */
        strcpy ((char *) insertStmt, (char *) insertDBStmts[1]);

        if (isUdoEnabled)
            strcat ((char *) insertStmt, (char *) udoItem[i]);
        else
            strcat ((char *) insertStmt, (char *) nonUdoItem[i]);

        /* Execute the INSERT statement */
        if(rc = GCIStmtExecDirect (svchp, insertStmt, errhp, 0, 0, 0, 0, GCI_DEFAULT) != GCI_SUCCESS)
        {
        SQLMSG;
			fprintf(stdout, "Error in Step 5 -- GCIStmtExecDirect: direct execute statement failed\nExiting!!\n");
			goto EXIT;
		}  
    }


    fprintf (stdout, "Step 5 done...Data inserted into the 'item' table\n");




    /* STEP 6.  Construct the INSERT statement for table 'orders'
    **          Insert data into the table 'orders'.
    */

    fprintf (stdout, "Populating table 'orders'\n");

    for (i = 0; i < NUM_ORDERS; i++)
    {
        fprintf (stdout, "Inserting row # %d\n", i+1);
        
        memset(insertStmt, 0, BUFFER_LEN);

        /* Construct the INSERT statement for table 'orders' */
        strcpy ((char *) insertStmt, (char *) insertDBStmts[2]);
        strcat ((char *) insertStmt, (char *) orders[i]);

        /* Execute the INSERT statement */
        if(rc = GCIStmtExecDirect (svchp, insertStmt, errhp, 0, 0, 0, 0, GCI_DEFAULT) != GCI_SUCCESS)
        {
        SQLMSG
			fprintf(stdout, "Error in Step 6 -- GCIStmtExecDirect: direct execute statement failed\nExiting!!\n");
			goto EXIT;
		}  
    }


    fprintf (stdout, "Step 6 done...Data inserted into the 'orders' table\n");



EXIT:

    /* CLEANUP: Close the statement handle
    **          Free the statement handle
    **          Disconnect from the datasource
    **          Free the connection and environment handles
    **          Exit
    */
    
    GCILogoff(svchp, errhp);

	if(svchp)
		GCIHandleFree((dvoid *)svchp, GCI_HTYPE_SVCCTX);
	
	if(errhp)
		GCIHandleFree((dvoid *)errhp, GCI_HTYPE_ERROR);
	
	if(envhp)
		GCIHandleFree((dvoid *)envhp, GCI_HTYPE_ENV);
	
    fprintf (stdout,"\n\nHit <Enter> to terminate the program...\n\n");
    in = getchar ();
    
    return (rc);
}
